#include "EnemyManager.h"
#include "Level.h"

EnemyManager::EnemyManager()
{
	enemyTexture.loadFromFile("Assets/Sprites/EnemySprites.png");	//Load the EnemySprites.png vector
}


EnemyManager::~EnemyManager()	//The destructor clears the contents of both the enemies and enemySpawns vectors
{
	for (int i = 0; i < enemies.size(); i++)
	{
		delete enemies[i];
	}

	for (int i = 0; i < enemySpawns.size(); i++)
	{
		delete enemySpawns[i];
	}
}

void EnemyManager::create(MySprite::SpriteType enemyType, sf::Vector2f spawn)
{
	enemySpawns.push_back(new SpawnStruct());	//Push back a new instance of SpawnStruct to enemySpawns
	enemySpawns[enemySpawns.size() - 1]->typeOfEnemy = enemyType;	//Set the typeOfEnemy at the new enemySpawns index to the enemyType parameter
	enemySpawns[enemySpawns.size() - 1]->spawnLocation = spawn;		//Set the spawnLocation at the new enemySpawns index to the spawn parameter
}

void EnemyManager::checkSpawn(Mario* mario)
{
	for (int i = 0; i < enemySpawns.size(); i++)
	{
		if (enemySpawns[i]->spawnLocation.x < mario->getPosition().x + 600 && !enemySpawns[i]->isSpawned)	//If we're near a spawn location and the enemy hasn't been spawned
		{
			enemySpawns[i]->isSpawned = true;	//We're spawning an enemy in, set isSpawned to true
			switch (enemySpawns[i]->typeOfEnemy)
			{
				case MySprite::SpriteType::GoombaSprite:	//If the enemy we're spawning is a goomba
					enemies.push_back(new Goomba());	//Push back a new goomba onto the enemies vector
					enemies[enemies.size() - 1]->setAlive(true);	//Set the goomba to be alive
					enemies[enemies.size() - 1]->setTexture(&enemyTexture);	//Set the texture to the sprite sheet
					enemies[enemies.size() - 1]->setPosition(enemySpawns[i]->spawnLocation);	//Set the goomba's position to be the spawn location
					break;

				case MySprite::SpriteType::KoopaSprite:	//If the enemy we're spawning is a koopa
					enemies.push_back(new Koopa());	//Push back a new koopa onto the enemies vector
					enemies[enemies.size() - 1]->setAlive(true);	//Set the koopa to be alive
					enemies[enemies.size() - 1]->setTexture(&enemyTexture);	//Set the texture to the sprite sheet
					enemies[enemies.size() - 1]->setPosition(enemySpawns[i]->spawnLocation);	//Set the koopa's position to be the spawn location
					break;
			}
		}

		else if (enemySpawns[i]->spawnLocation.x < mario->getPosition().x - 560 && enemySpawns[i]->isSpawned)	//If the enemy's position is behind mario and off the screen (-560) and is spawned
		{
			enemySpawns.erase(enemySpawns.begin() + i);	//Delete the spawn information for this enemy as it is no longer needed
		}
	}
}

void EnemyManager::update(float dt, Mario* mario, ItemManager* itemManager)
{
	Goomba* thisGoomba;
	Koopa* thisKoopa;
	std::vector<GameItems*> items = itemManager->getItems();	//Call getItems to get all the current game items

	for (int i = 0; i < enemies.size(); i++)
	{
		if (enemies[i]->isDying() && Timer::setWaitTime(enemies[i]->getDeathTime(), dt))	//If the enemy is dying (playing death animation) and the time taken (getDeathTime) has elapsed
		{
			enemies[i]->setAlive(false);	//Set isAlive to false to stop drawing the enemy
		}

		if (mario->isAlive())
		{
			if (enemies[i]->isAlive())
			{
				if (Collision::checkBoundingBox(mario, enemies[i]) && !mario->getInvincible() && !enemies[i]->getInvincible() && !enemies[i]->isDying())	//If the enemy collides with Mario, neither are invincible and the enemy isn't already dying
				{
					if (!mario->handleEnemyCollision(enemies[i]))	//If Mario has hit the enemy, not killed it (i.e. returned false)
					{
						mario->powerDown();	//We power Mario down
					}

					else    //Otherwise, he has killed the enemy (returned true)
					{
						switch (enemies[i]->getSpriteType())
						{
							case MySprite::SpriteType::GoombaSprite:
								mario->addScore(100);	//Add 100 score for killing a goomba
								mario->bounce();
								thisGoomba = (Goomba*)enemies[i];	//Cast enemies[i] to a Goomba pointer to access Goomba specific functions
								thisGoomba->kill();	//Call kill to kill thisGoomba
								break;

							case MySprite::SpriteType::KoopaSprite:
								thisKoopa = (Koopa*)enemies[i];	//Cast enemies[i] to a Koopa pointer to access Koopa specific functions
								thisKoopa->handleMarioCollision(mario);	//Get thisKoopa to handleMarioCollision, passing the mario pointer as an argument
								break;
						}
					}
				}

				else
				{
					for (int j = 0; j < items.size(); j++)	//For every item
					{
						if (Collision::checkBoundingBox(enemies[i], items[j]))	//If an enemy collides with an item
						{
							if (items[j]->isAlive())
							{
								if (items[j]->getSpriteType() == MySprite::SpriteType::PipeSprite)	//If it's a pipe
								{
									enemies[i]->setVelocity(-(enemies[i]->getVelocity().x), enemies[i]->getVelocity().y);	//It will definitely be the pipe edge, so invert the x velocity

									if (enemies[i]->getSpriteType() == MySprite::SpriteType::KoopaSprite)	//If the enemy is a Koopa
									{
										thisKoopa = (Koopa*)enemies[i];	//Cast enemies[i] to a Koopa pointer to access Koopa specific functions
											
										if (thisKoopa->getKoopaState() == Koopa::KoopaState::MovingShell)	//If the Koopa is in its shell and is moving
										{
											Level::audioManager->playSoundByName("Collide");	//Play the collide sound to warn the player audibly that the shell has reversed its direction
										}
									}
								}

								else if (items[j]->getSpriteType() == MySprite::SpriteType::BlockSprite)	//If it's a block
								{
									items[j]->handleCollision(enemies[i], dt);	//Get the block to handle the enemy collision

									if (Collision::checkBoundingBox(items[j], mario))
									{
										if (mario->getVelocity().y <= 0.f && (mario->getPosition().y >= items[j]->getPosition().y + (items[j]->getSize().y * 0.75)) && mario->getPosition().x + mario->getSize().x > items[j]->getPosition().x && mario->getPosition().x < items[j]->getPosition().x + items[j]->getSize().x)	//If we're going up and we're underneath both block axis
										{
											enemies[i]->setAlive(false);
										}
									}
								}
							}
						}
					}

					if (enemies[i]->getSpriteType() == MySprite::SpriteType::KoopaSprite)	//If enemies[i] is a Koopa
					{
						thisKoopa = (Koopa*)enemies[i];	//Cast enemies[i] to a Koopa pointer to access Koopa specific functions

						if (thisKoopa->getKoopaState() == Koopa::KoopaState::MovingShell)	//If the Koopa is moving in its shell
						{
							for (int k = 0; k < enemies.size(); k++)	//For every enemy
							{
								if (Collision::checkBoundingBox(enemies[k], thisKoopa) && k != i)	//If it collides with any enemy, not itself (k != i)
								{
									enemies[k]->setAlive(false);	//Set isAlive for that enemy to false
									enemies.erase(enemies.begin() + k);	//Erase the enemy from the enemies vector
									Level::audioManager->playSoundByName("Squish");
								}
							}
						}
					}
				}

				enemies[i]->update(dt);
			}
		}
	}

	checkSpawn(mario);	//Check if we should spawn in any enemies
	deathCheck(mario);	//Check if we should kill any enemies
}

void EnemyManager::deathCheck(Mario* mario)
{
	for (int i = 0; i < enemies.size(); i++)
	{
		if (enemies[i]->isAlive())
		{
			if (enemies[i]->getPosition().y >= 768)	//If the enemy has fallen below the map (into a pit)
			{
				enemies[i]->setAlive(false);	//Set isAlive to false
				enemies.erase(enemies.begin() + i);	//Erase this enemy from the enemies vector
			}

			else if ((enemies[i]->getPosition().x < mario->getPosition().x - 512 - enemies[i]->getSize().x) && mario->getPosition().x > 512 + enemies[i]->getSize().x)
			{
				enemies[i]->setAlive(false);	//Set isAlive to false
				enemies.erase(enemies.begin() + i);	//Erase this enemy from the enemies vector
			}

			else if (enemies[i]->getPosition().x > mario->getPosition().x + 2050)
			{
				enemies[i]->setAlive(false);	//Set isAlive to false
				enemies.erase(enemies.begin() + i);	//Erase this enemy from the enemies vector
			}
		}

		else if (!enemies[i]->isAlive() && enemies[i]->isDying())	//If the enemy is dead and is performing its death animation
		{
			enemies.erase(enemies.begin() + i);	//Erase this enemy from the enemies vector
		}
	}
}

void EnemyManager::render(sf::RenderWindow* window)
{
	for (int i = 0; i < enemies.size(); i++)
	{
		if (enemies[i]->isAlive() || enemies[i]->isDying())	//If the enemy is alive or is performing its death animation
		{
			window->draw(*enemies[i]);	//Draw it to the screen
		}
	}
}